﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.Operations;
using BMS.VistaIntegration.Data;
using BMS.Utils;
using System.Threading.Tasks;
using BMS.VistaWorker2;
using BMS.VistaIntegration.Exceptions;
using BMS.VistaWorker2.Writer;
using BMS.VistaIntegration.Operations.Concrete;
using BMS.VistaIntegration.VistA;
using BMS.VistaIntegration.Audit;
using BMS.VistaIntegration.Managers;
using BMS.ServicesWrapper.BMService;

namespace BMS.VistaIntegration.Commands
{
    public class RunOnDemandCommand
    {
        public bool Succes { get; private set; }
        public string ErrorMessage { get; private set; }

        private readonly Crawler crawler;
        private readonly BmsLogger Logger = new BmsLogger("VistaIntegration RunOnDemandCommand Message:");
        private readonly IVistASessionFactory sessionFactory;

        public RunOnDemandCommand(Crawler crawler, IVistASessionFactory sessionFactory)
        {
            this.sessionFactory = sessionFactory;
            this.crawler = crawler;
        }

        private IList<Parameter> GetParameters(VistaDataType types, DateTime? startDate, DateTime? endDate)
        {
            List<Parameter> parameters = Utilities.GetVistaTypes(types)
                .Select(s => new Parameter(s, startDate, endDate))
                .ToList();
            return parameters;
        }

        public void Run(VistASite vistASite, VistAConnectionInfo connection, VistaDataType types, DateTime? startDate, DateTime? endDate)
        {
            List<VistaDataType> dataTypes = Utilities.GetVistaTypes(types).ToList();
            foreach (VistaDataType dt in dataTypes)
            {
                var parameters = GetParameters(dt, startDate, endDate);
                BaseOperation operation = new Operation(vistASite, parameters);
                
                Task task = new Task(() => StartRunOnDemandTaskAction(operation, vistASite, connection));                
                task.ContinueWith((t) => { crawler.OperationManager.EndRunTask(vistASite.Id, t); t.Dispose(); });
                crawler.OperationManager.RunOperation(vistASite.Id, task, dt);
            }
            Succes = true;            
        }

        private BMS.DataContracts.JobLogInfo MakeJobLog(Parameter param, VistASite site)
        {
            BMS.DataContracts.JobLogInfo jobLog = AuditUtilities.MakeJobLog(param, site, sessionFactory.RetrievalMethod);
            jobLog.LaunchType = JobLaunchType.Manual;
            return jobLog;
        }

        private void OnJobLogFaild(VistASite site, BMS.DataContracts.JobLogInfo jobLog, Exception e)
        {
            try
            {
                AuditUtilities.OperationFailed(site, jobLog, e.Message, e.ToString());
            }
            catch (Exception ex)
            {
                string msj = string.Format("Exception on update log failed for job id {0} :\n{1}", jobLog.Id, ex);
                Logger.LogError(msj);
            }
        }

        private void RunOperation(BaseOperation operation, VistASite vistASite, VistAConnectionInfo connectionInfo)
        {
            using (IVistASession session = sessionFactory.MakeVistASession(vistASite))
            {
                session.Open(connectionInfo);
                operation.Run(session, crawler.WriterManagerFactory, crawler.CancellationTokenSource.Token);
            }
        }

        private void UpdateAuditData(VistASite site, ParameterEventArgs eventArgs)
        {
            BMS.DataContracts.JobLogInfo jobLog = MakeJobLog(eventArgs.Parameter, site);
            AuditUtilities.OperationUpdated(site, jobLog, eventArgs);
        }

        private void StartRunOnDemandTaskAction(BaseOperation operation, VistASite vistASite, VistAConnectionInfo connectionInfo)
        {
            try
            {
                foreach (Parameter param in operation.Parameters)
                    param.JobLaunchType = JobLaunchType.Manual;
                operation.TypeUpdating += (s, e) => UpdateAuditData(vistASite, e);
                operation.TypeUpdated += (s, e) => UpdateAuditData(vistASite, e);
                RunOperation(operation, vistASite, connectionInfo);
            }
            catch (Exception e)
            {
                string msj = string.Format("Run on demand task exception for {0} :\n{1}", operation, e);
                Logger.LogError(msj);
                IDictionary<Parameter, BMS.DataContracts.JobLogInfo> logDictionary = operation.Parameters.ToDictionary(p => p, p => MakeJobLog(p, vistASite));
                if (logDictionary != null)
                {
                    logDictionary.Values.Where(s => s.Status == JobStatus.Running).ForEach(j => OnJobLogFaild(vistASite, j, e));
                }
            }
        }
    }
}
